// Engine.cpp: implementation of the Engine class.
//
//////////////////////////////////////////////////////////////////////
#include "Move.h"
#include "stdafx.h"
#include "Engine.h"

#include <iostream>
#include <time.h>
using namespace std;


//////////////////////////////////////////////////////////////////////
// Engine 
//////////////////////////////////////////////////////////////////////

#define INVALID_SQUARE 65

//Pieces constants
#define PAWN 0
#define KNIGHT 1
#define BISHOP 2
#define ROCK 3
#define QUEEN 4
#define KING 5

#define WHITE 0
#define BLACK 1
#define EMPTY 2

//Castle constants
#define INITIAL_WHITE_KING_SQUARE 4
#define INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE 7
#define INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE 0
#define INITIAL_BLACK_KING_SQUARE 60
#define INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE 63
#define INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE 56

#define WHITE_SHORT_CASTLE_PASS1 5
#define WHITE_SHORT_CASTLE_PASS2 6

#define WHITE_LONG_CASTLE_PASS1 3
#define WHITE_LONG_CASTLE_PASS2 2
#define WHITE_LONG_CASTLE_PASS3 1

#define BLACK_SHORT_CASTLE_PASS1 61
#define BLACK_SHORT_CASTLE_PASS2 62

#define BLACK_LONG_CASTLE_PASS1 59
#define BLACK_LONG_CASTLE_PASS2 58
#define BLACK_LONG_CASTLE_PASS3 57

#define WHITE_SHORT_CASTLE 0
#define BLACK_SHORT_CASTLE 1
#define WHITE_LONG_CASTLE 2
#define BLACK_LONG_CASTLE 3

//Move generation constants
#define NON_CAPTURE_MOVE_COUNT 100
#define CAPTURE_MOVE_COUNT 35
#define PROMOTE_MOVE_COUNT 8
#define PROMOTE_AND_CAPTURE_MOVE_COUNT 8

//Search constants
#define INFINITE_NEGATIVE -90000 //- 10 kings
#define INFINITE_POSITIVE 90000 // 10 kings
#define MAX_QPLY 6


//// GAME STAGE CONSTS /////////////////
#define OPENING_MOVES 15
#define ENDING_NO_PAWN_SCORE 1200

//// PIECE VALUES //////////////////////
#define PAWN_VALUE 100
#define KNIGHT_VALUE 330
#define BISHOP_VALUE 330
#define ROCK_VALUE 520
#define QUEEN_VALUE 980
#define KING_VALUE 9800 //10 queen
////////////////////////////////////////


//// PAWN CALCULATE CONST //////////////
#define DOUBLED_PAWN_PENALTY -12 
#define BACKWARD_PAWN_PENALTY -6
#define BACKWARD_PAWN_ATTACKED_PENALTY -4
#define BACKWARD_PAWN_IN_HALF_OPEN_FILE_PENALTY -4
#define BLOCKED_D_E_PAWN_PENALTY -15
#define PAWN_CLOSED_TO_KING_BONUS 10
#define KNIGHT_NEAR_OPPONENT_PIECE_BONUS_OPE 2
#define KNIGHT_NEAR_OPPONENT_PIECE_BONUS_END 4
#define KNIGHT_FAR_FROM_KING_PENALTY -1
#define KNIGHT_SECURE_FROM_ENEMY_PAWNS_BONUS 8
#define BISHOP_ATTACK_MOST_VALUED_BONUS 8
#define BISHOP_EXTRA_MATERIAL_IN_ENDING 10
#define BISHOP_ATTACK_ADYACENT_ENEMY_KING_BONUS 5
#define ROCK_ATTACK_MOST_VALUED_BONUS 8
#define ROCK_ON_HALF_OPEN_FILE_BONUS 10
#define ROCK_ON_FULL_OPEN_FILE_EXTRA_BONUS 4
#define ROCK_TAXICAB_PENALTY -2
#define ROCK_ON_SEVENTH_COLUMN_BONUS 10

#define EMPTY_SQUARE_ATTACK_BONUS 1

#define QUEEN_TAXICAB_PENALTY -2

#define KING_IN_OPEN_FILE_PENALTY 23
#define KING_WITH_NO_PAWNS_PENALTY -8
#define KING_CASTLED_BONUS 10
#define KING_MOVED_BEFORE_CASTLING_PENALTY -40



//Game status constants
#define BLACK_CHECKMATED 2
#define WHITE_CHECKMATED 1
#define DRAW 3
#define WHITE_RESIGN 4
#define BLACK_RESIGN 5
#define ON_PLAYING 0
#define CHECKMATE_FOUND 0


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Engine::Engine()
{
	InitializeGlobalVariables();
	GenerateColumnUpMoves();
	GenerateDiagonal45Moves();
	GenerateRankRightMoves();
	GenerateDiagonal135Moves();
	GenerateColumnDownMoves();
	GenerateDiagonal225Moves();
	GenerateRankLeftMoves();
	GenerateDiagonal315Moves();
	GenerateKingMoves();
	GenerateKnightMoves();
	InitializeBoard();
}

Engine::~Engine()
{
	
}

//////////////////////////////////////////////////////////////////////
// Pre-Porcessed move generation
//////////////////////////////////////////////////////////////////////

void Engine::InitializeGlobalVariables()
{
	int rankPivot=0, columnPivot=0;

	//Initialize rankNumbers and columnNumbers arrays.
	for (int square=0; square<64; square++)
	{
		rankNumbers[square] = rankPivot;
		columnNumbers[square] = columnPivot;
		squaresByColumn[square] = columnPivot * 8 + rankPivot;

		//array used to clear the attacks array.
		blankAttackSquares[square] = 0;
	
		columnPivot++;

		if (columnPivot == 8)
		{
			rankPivot++;
			columnPivot = 0;
		}	
	}
	
	//Initialize pieceVal array
	pieceValue[PAWN] = PAWN_VALUE;
	pieceValue[KNIGHT] = KNIGHT_VALUE;
	pieceValue[BISHOP] = BISHOP_VALUE;
	pieceValue[ROCK] = ROCK_VALUE;
	pieceValue[QUEEN] = QUEEN_VALUE;
	pieceValue[KING] = KING_VALUE;

	isolatedPawnPenalty[0] = -12;
	isolatedPawnPenalty[1] = -14;
	isolatedPawnPenalty[2] = -16;
	isolatedPawnPenalty[3] = -20;
	isolatedPawnPenalty[4] = -20;
	isolatedPawnPenalty[5] = -16;
	isolatedPawnPenalty[6] = -14;
	isolatedPawnPenalty[7] = -12;
}

void Engine::GenerateColumnUpMoves ()
{
	int posibleSquares;
	int rank;
	for (int square=0; square<64; square++)
    {
        rank = rankNumbers[square];
      	posibleSquares=0;
		for (int i=0; i<7-rank; i++)
		{
			 columnUpMoves[square][i] = square + 8 * (i + 1);
			 posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			//Fill left direction positions whit invalid square 
			columnUpMoves[square][j] = INVALID_SQUARE;
		}
		
    }
    return;
}

void Engine::GenerateColumnDownMoves()
{
	int posibleSquares;
	int rank;
    for (int square=0; square<64; square++)
    {
        rank = rankNumbers[square];
        posibleSquares=0;
	    for (int i=0; i<rank; i++)
		{
            columnDownMoves[square][i] = square - 8 * (i + 1);
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			columnDownMoves[square][j] = INVALID_SQUARE;
		}
    }
}

void Engine::GenerateRankRightMoves()
{
	int column;
	int posibleSquares;
    for (int square=0; square<64; square++)
    {
		column = columnNumbers[square];
		posibleSquares=0;
        for (int i=0; i<7-column; i++)
		{
            rankRightMoves[square][i] = square + i + 1;
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			rankRightMoves[square][j] = INVALID_SQUARE;
		}
    }
}

void Engine::GenerateRankLeftMoves()
{
	int column;
	int posibleSquares;
    for (int square=0; square<64; square++)
    {
		column = columnNumbers[square];
		posibleSquares=0;
        for (int i=0; i<column; i++)
		{
            rankLeftMoves[square][i] = square - i - 1;
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			rankLeftMoves[square][j] = INVALID_SQUARE;
		}
    }
}

void Engine::GenerateDiagonal45Moves()
{
	int column;
	int rank;
	int posibleSquares;
	int movTo;
	for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];
		posibleSquares=0;
		if ((7-column)<(7-rank))
			movTo = 7-column;
		else
			movTo = 7-rank;
      
        for (int i=0; i<movTo; i++)
		{
            diagonal45Moves[square][i] = square + 9 * (i + 1);
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			diagonal45Moves[square][j] = INVALID_SQUARE;
		}
    }
}
void Engine::GenerateDiagonal135Moves()
{
	int column;
	int rank;
	int posibleSquares;
	int movTo;
	for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];
		posibleSquares=0;
		if ((7-column)<rank)
			movTo = 7-column;
		else
			movTo = rank;
      
        for (int i=0; i<movTo; i++)
		{
            diagonal135Moves[square][i] = square - 7 * (i + 1);
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			diagonal135Moves[square][j] = INVALID_SQUARE;
		}
    }
}
void Engine::GenerateDiagonal225Moves()
{
	int column;
	int rank;
	int posibleSquares;
	int movTo;
	for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];
		posibleSquares=0;
		if (column<rank)
			movTo = column;
		else
			movTo = rank;
      
        for (int i=0; i<movTo; i++)
		{
            diagonal225Moves[square][i] = square - 9 * (i + 1);
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			diagonal225Moves[square][j] = INVALID_SQUARE;
		}
    }
}
void Engine::GenerateDiagonal315Moves()
{
	int column;
	int rank;
	int posibleSquares;
	int movTo;
	for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];
		posibleSquares=0;
		if (column<(7-rank))
			movTo = column;
		else
			movTo = 7-rank;
      
        for (int i=0; i<movTo; i++)
		{
            diagonal315Moves[square][i] = square + 7 * (i + 1);
			posibleSquares++;
		}
		for (int j=7; j>=posibleSquares; j--)
		{
			diagonal315Moves[square][j] = INVALID_SQUARE;
		}
    }
}

void Engine::GenerateKingMoves()
{
	int column;
	int rank;
    for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];

        if (square == 0)
		{
			kingMoves[square][0] = 1; kingMoves[square][1] = 9; kingMoves[square][2] = 8;
			kingMoves[square][3] = INVALID_SQUARE; kingMoves[square][4] = INVALID_SQUARE; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (square == 7)
		{
			kingMoves[square][0] = 6; kingMoves[square][1] = 14; kingMoves[square][2] = 15;
			kingMoves[square][3] = INVALID_SQUARE; kingMoves[square][4] = INVALID_SQUARE; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (square == 56)
		{
			kingMoves[square][0] = 48; kingMoves[square][1] = 49; kingMoves[square][2] = 57;
			kingMoves[square][3] = INVALID_SQUARE; kingMoves[square][4] = INVALID_SQUARE; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (square == 63)
		{
			kingMoves[square][0] = 62; kingMoves[square][1] = 54; kingMoves[square][2] = 55;
			kingMoves[square][3] = INVALID_SQUARE; kingMoves[square][4] = INVALID_SQUARE; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank == 0)
		{
			kingMoves[square][0] = square-1; kingMoves[square][1] = square+7; kingMoves[square][2] = square+8;
			kingMoves[square][3] = square+9; kingMoves[square][4] = square+1; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank == 7)
		{
			kingMoves[square][0] = square-1; kingMoves[square][1] = square-9; kingMoves[square][2] = square-8;
			kingMoves[square][3] = square-7; kingMoves[square][4] = square+1; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (column == 0)
		{
			kingMoves[square][0] = square+8; kingMoves[square][1] = square+9; kingMoves[square][2] = square+1;
			kingMoves[square][3] = square-7; kingMoves[square][4] = square-8; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else if (column == 7)
		{
			kingMoves[square][0] = square-8; kingMoves[square][1] = square-9; kingMoves[square][2] = square-1;
			kingMoves[square][3] = square+7; kingMoves[square][4] = square+8; kingMoves[square][5] = INVALID_SQUARE;
			kingMoves[square][6] = INVALID_SQUARE;kingMoves[square][7] = INVALID_SQUARE;
		}
		else
		{
			kingMoves[square][0] = square+8; kingMoves[square][1] = square+9; kingMoves[square][2] = square+1;
			kingMoves[square][3] = square-7; kingMoves[square][4] = square-8; kingMoves[square][5] = square-9;
			kingMoves[square][6] = square-1; kingMoves[square][7] = square+7;
		}
	}    
}

void Engine::GenerateKnightMoves()
{
	int column;
	int rank;
    for (int square=0; square<64; square++)
    {
        column = columnNumbers[square];
        rank = rankNumbers[square];

		if (column > 1 && column < 6 && rank > 1 && rank < 6)
		{
			knightMoves[square][0] = square + 17;knightMoves[square][1] = square + 10;
			knightMoves[square][2] = square - 6; knightMoves[square][3] = square - 15;
			knightMoves[square][4] = square - 17;knightMoves[square][5] = square - 10;
			knightMoves[square][6] = square + 6; knightMoves[square][7] = square + 15;
		}
		else if (square==0)
		{
			knightMoves[square][0] = 17; knightMoves[square][1] = 10;
			knightMoves[square][2] = INVALID_SQUARE; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==1)
		{
			knightMoves[square][0] = 16; knightMoves[square][1] = 18;
			knightMoves[square][2] = 11; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==6)
		{
			knightMoves[square][0] = 12; knightMoves[square][1] = 21;
			knightMoves[square][2] = 23; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==7)
		{
			knightMoves[square][0] = 22; knightMoves[square][1] = 13;
			knightMoves[square][2] = INVALID_SQUARE; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==8)
		{
			knightMoves[square][0] = 25; knightMoves[square][1] = 18;
			knightMoves[square][2] = 1; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==9)
		{
			knightMoves[square][0] = 24; knightMoves[square][1] = 26;
			knightMoves[square][2] = 19; knightMoves[square][3] = 3;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==14)
		{
			knightMoves[square][0] = 31; knightMoves[square][1] = 29;
			knightMoves[square][2] = 20; knightMoves[square][3] = 4;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==15)
		{
			knightMoves[square][0] = 30; knightMoves[square][1] = 21;
			knightMoves[square][2] = 5; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==48)
		{
			knightMoves[square][0] = 58; knightMoves[square][1] = 42;
			knightMoves[square][2] = 33; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==49)
		{
			knightMoves[square][0] = 59; knightMoves[square][1] = 43;
			knightMoves[square][2] = 34; knightMoves[square][3] = 32;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==54)
		{
			knightMoves[square][0] = 60; knightMoves[square][1] = 44;
			knightMoves[square][2] = 37; knightMoves[square][3] = 39;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==55)
		{
			knightMoves[square][0] = 61; knightMoves[square][1] = 45;
			knightMoves[square][2] = 38; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==56)
		{
			knightMoves[square][0] = 50; knightMoves[square][1] = 41;
			knightMoves[square][2] = INVALID_SQUARE; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==57)
		{
			knightMoves[square][0] = 40; knightMoves[square][1] = 42;
			knightMoves[square][2] = 51; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==62)
		{
			knightMoves[square][0] = 52; knightMoves[square][1] = 45;
			knightMoves[square][2] = 47; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (square==63)
		{
			knightMoves[square][0] = 53; knightMoves[square][1] = 46;
			knightMoves[square][2] = INVALID_SQUARE; knightMoves[square][3] = INVALID_SQUARE;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank==0)
		{
			knightMoves[square][0] = square + 6; knightMoves[square][1] = square + 15;
			knightMoves[square][2] = square + 17; knightMoves[square][3] = square + 10;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank==1)
		{
			knightMoves[square][0] = square - 10; knightMoves[square][1] = square + 6;
			knightMoves[square][2] = square + 15; knightMoves[square][3] = square + 17;
			knightMoves[square][4] = square + 10; knightMoves[square][5] = square - 6;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank==6)
		{
			knightMoves[square][0] = square - 17; knightMoves[square][1] = square -10;
			knightMoves[square][2] = square + 6; knightMoves[square][3] = square +10;
			knightMoves[square][4] = square - 6; knightMoves[square][5] = square - 15;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (rank==7)
		{
			knightMoves[square][0] = square - 17; knightMoves[square][1] = square -10;
			knightMoves[square][2] = square - 6; knightMoves[square][3] = square - 15;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (column==0)
		{
			knightMoves[square][0] = square-15; knightMoves[square][1] = square-6;
			knightMoves[square][2] = square+10; knightMoves[square][3] = square+17;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (column==1)
		{
			knightMoves[square][0] = square-17; knightMoves[square][1] = square-15;
			knightMoves[square][2] = square-6; knightMoves[square][3] = square+10;
			knightMoves[square][4] = square+17; knightMoves[square][5] = square+15;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else if (column==6)
		{
			knightMoves[square][0] = square-15; knightMoves[square][1] = square-17;
			knightMoves[square][2] = square-10; knightMoves[square][3] = square+6;
			knightMoves[square][4] = square+15; knightMoves[square][5] = square+17;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
		else
		{
			knightMoves[square][0] = square-17; knightMoves[square][1] = square-10;
			knightMoves[square][2] = square+6; knightMoves[square][3] = square+15;
			knightMoves[square][4] = INVALID_SQUARE; knightMoves[square][5] = INVALID_SQUARE;
			knightMoves[square][6] = INVALID_SQUARE; knightMoves[square][7] = INVALID_SQUARE;
		}
	}
}

void Engine::GenerateNoPawnDirections(int slidingPiece, int square)
{
	if (slidingPiece == KNIGHT)
		memcpy((void*)oneDirection, (void*)knightMoves[square], sizeof(int)*8);
	else if (slidingPiece == BISHOP)
	{
		memcpy((void*)fourDirections[0], (void*)diagonal45Moves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[1], (void*)diagonal135Moves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[2], (void*)diagonal225Moves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[3], (void*)diagonal315Moves[square], sizeof(int)*8);
	}
	else if (slidingPiece == ROCK)
	{
		memcpy((void*)fourDirections[0], (void*)columnUpMoves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[1], (void*)rankRightMoves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[2], (void*)columnDownMoves[square], sizeof(int)*8);
		memcpy((void*)fourDirections[3], (void*)rankLeftMoves[square], sizeof(int)*8);
	}
	else if (slidingPiece == QUEEN)
	{
		memcpy((void*)eightDirections[0], (void*)columnUpMoves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[1], (void*)diagonal45Moves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[2], (void*)rankRightMoves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[3], (void*)diagonal135Moves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[4], (void*)columnDownMoves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[5], (void*)diagonal225Moves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[6], (void*)rankLeftMoves[square], sizeof(int)*8);
		memcpy((void*)eightDirections[7], (void*)diagonal315Moves[square], sizeof(int)*8);
	}
	else if (slidingPiece == KING)
		memcpy((void*)oneDirection, (void*)kingMoves[square], sizeof(int)*8);
}

//////////////////////////////////////////////////////////////////////
// Board - Move generation
//////////////////////////////////////////////////////////////////////

void Engine::InitializeBoard()
{
	//Square A1 is 0 position and square H8 is 63 position
	int i;
	
	for (i=0; i<=15; i++)
        squareSide[i] = WHITE;
    for (i=16; i<=47; i++)
        squareSide[i] = EMPTY;
    for (i = 48; i <= 63; i++)
        squareSide[i] = BLACK;

    squarePiece[0] = ROCK;
    squarePiece[1] = KNIGHT;
    squarePiece[2] = BISHOP;
    squarePiece[3] = QUEEN;
    squarePiece[4] = KING;
    squarePiece[5] = BISHOP;
    squarePiece[6] = KNIGHT;
    squarePiece[7] = ROCK;

	//Add pawns in ALL nopawns squares only fot clean garbage;
    for (i = 8; i <= 55; i++)
        squarePiece[i] = PAWN;
	
    squarePiece[56] = ROCK;
    squarePiece[57] = KNIGHT;
    squarePiece[58] = BISHOP;
    squarePiece[59] = QUEEN;
    squarePiece[60] = KING;
    squarePiece[61] = BISHOP;
    squarePiece[62] = KNIGHT;
    squarePiece[63] = ROCK;

    sideToMove = WHITE;

	//!! Jaque matye en uno
	/*
	squarePiece[17] = BISHOP;
	squarePiece[21] = QUEEN;
	squareSide[17] = WHITE;
	squareSide[21] = WHITE;
	*/

	//!! Jaque matye en dos
	/*
	squarePiece[25] = BISHOP;
	squareSide[25] = WHITE;
	squarePiece[23] = QUEEN;
	squareSide[23] = WHITE;

	squarePiece[46] = BISHOP;
	squareSide[46] = BLACK;

	squareSide[52] = EMPTY;
	squareSide[61] = EMPTY;
	squareSide[62] = EMPTY;
	squareSide[63] = EMPTY;
	squareSide[55] = EMPTY;
	*/

	//!! Jaque matye en tres
	/*
	squarePiece[23] = QUEEN;
	squareSide[23] = WHITE;

	squarePiece[18] = ROCK;
	squareSide[18] = WHITE;

	squarePiece[26] = ROCK;
	squareSide[26] = WHITE;

	squarePiece[43] = BISHOP;
	squareSide[43] = BLACK;

	squareSide[52] = EMPTY;
	squareSide[61] = EMPTY;
	squareSide[62] = EMPTY;
	squareSide[63] = EMPTY;
	squareSide[55] = EMPTY;
	*/

    whiteKingSquare = 4;
    blackKingSquare = 60;
	whiteKingCastled = false;
	blackKingCastled = false;

	GameStatus = ON_PLAYING;
}

void Engine::GenerateAllMoves()
{
	//Clear previus arrays values. 
	//Te trick is copy blank moves in the directions of memory where previous moves are stored.
	memcpy((void*)legalNonCapture, (void*)blankLegalNonCapture, sizeof(Move)*NON_CAPTURE_MOVE_COUNT);
	memcpy((void*)legalCapture, (void*)blankLegalCapture, sizeof(Move)*CAPTURE_MOVE_COUNT);
	memcpy((void*)legalPromote, (void*)blankLegalPromote, sizeof(Move)*PROMOTE_MOVE_COUNT);
	memcpy((void*)legalPromoteAndCapture, (void*)blankLegalPromoteAndCapture, sizeof(Move)*PROMOTE_AND_CAPTURE_MOVE_COUNT);

	legalNonCaptureCount = 0;
    legalCaptureCount = 0;
    legalPromoteCount = 0;
    legalPromoteAndCaptureCount = 0;

	int column;

	int movingPiece;
	int pawnFrontSquare;
	int pawnFirstMoveSquare;
	int pawnCaptureLeftSquare;
	int pawnCaptureRightSquare;

	int direction;
	int directionSquare;
	int moveSquare;
	int moveSquareSide;

	legalNonCapture[0];

	for (int square=0; square<64; square++)
    {
		if (sideToMove == squareSide[square])
        {
			movingPiece = squarePiece[square];
			column = columnNumbers[square];
		
			if (movingPiece == PAWN)
            {
				if (sideToMove == WHITE)
                {
					pawnFrontSquare = square + 8;
					if (squareSide[pawnFrontSquare] == EMPTY)
					{
						TestAndAddMove(square, pawnFrontSquare, PAWN, false, -1);
						if (square <= 15)
						{
							pawnFirstMoveSquare = square + 16;
							if (squareSide[pawnFirstMoveSquare] == EMPTY)
								TestAndAddMove(square, pawnFirstMoveSquare, PAWN, false, -1);
						}
					}

					//Camtures
                    pawnCaptureLeftSquare = square + 7;
                    if (column != 0 && squareSide[pawnCaptureLeftSquare] == BLACK)
                        TestAndAddMove(square, pawnCaptureLeftSquare, PAWN, true, -1);

                    pawnCaptureRightSquare = square + 9;
                    if (column != 7 && squareSide[pawnCaptureRightSquare] == BLACK)
                        TestAndAddMove(square, pawnCaptureRightSquare, PAWN, true, -1);
				}
				else
				{
					pawnFrontSquare = square - 8;
					if (squareSide[pawnFrontSquare] == EMPTY)
					{
						TestAndAddMove(square, pawnFrontSquare, PAWN, false, -1);
						if (square >= 48)
						{
							pawnFirstMoveSquare = square - 16;
							if (squareSide[pawnFirstMoveSquare] == EMPTY)
								TestAndAddMove(square, pawnFirstMoveSquare, PAWN, false, -1);
						}
					}

					//Camtures
                    pawnCaptureLeftSquare = square - 7;
                    if (column != 7 && squareSide[pawnCaptureLeftSquare] == WHITE)
                        TestAndAddMove(square, pawnCaptureLeftSquare, PAWN, true, -1);

                    pawnCaptureRightSquare = square - 9;
                    if (column != 0 && squareSide[pawnCaptureRightSquare] == WHITE)
                        TestAndAddMove(square, pawnCaptureRightSquare, PAWN, true, -1);
				}
			}
			else
			{
				
				GenerateNoPawnDirections(movingPiece, square);
				if (movingPiece == ROCK || movingPiece == BISHOP)
				{
					for (direction=0; direction<4; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = fourDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							//se deslizo a un casillero vacio
							else if (moveSquareSide == EMPTY) 
								TestAndAddMove(square, moveSquare, movingPiece, false, -1);
							//capturo una pieza
							else 
							{
								TestAndAddMove(square, moveSquare, movingPiece, true, -1);
								break;
							}
						}
					}
				}
				else if (movingPiece == KING)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						
						//Los casilleros de piezas amigas son ignorados completamente
						//pero el testeo sigue en el resto de los casilleros
						if (moveSquareSide == EMPTY) 
							TestAndAddMove(square, moveSquare, movingPiece, false, -1);
						//capturo una pieza
						else if (moveSquareSide != sideToMove)
							TestAndAddMove(square, moveSquare, movingPiece, true, -1);
					}

					//Castle
					if (!whiteKingCastled && sideToMove == WHITE)
					{
						if (square == INITIAL_WHITE_KING_SQUARE &&
							squarePiece[INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE] == ROCK &&
							squareSide[INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE] == WHITE &&
							squareSide[WHITE_SHORT_CASTLE_PASS1] == EMPTY &&
							squareSide[WHITE_SHORT_CASTLE_PASS2] == EMPTY &&
							!IsSquareInAttack(INITIAL_WHITE_KING_SQUARE, WHITE) &&
							!IsSquareInAttack(WHITE_SHORT_CASTLE_PASS1, WHITE) &&
							!IsSquareInAttack(WHITE_SHORT_CASTLE_PASS2, WHITE))
							TestAndAddMove(square, WHITE_SHORT_CASTLE_PASS2, movingPiece, false, WHITE_SHORT_CASTLE);

						if (square == INITIAL_WHITE_KING_SQUARE &&
							squarePiece[INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE] == ROCK &&
							squareSide[INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE] == WHITE &&
							squareSide[WHITE_LONG_CASTLE_PASS1] == EMPTY &&
							squareSide[WHITE_LONG_CASTLE_PASS2] == EMPTY &&
							squareSide[WHITE_LONG_CASTLE_PASS3] == EMPTY &&
							!IsSquareInAttack(INITIAL_WHITE_KING_SQUARE, WHITE) &&
							!IsSquareInAttack(WHITE_LONG_CASTLE_PASS1, WHITE) &&
							!IsSquareInAttack(WHITE_LONG_CASTLE_PASS2, WHITE)) //PASS3 is not tested for check
							TestAndAddMove(square, WHITE_LONG_CASTLE_PASS2, movingPiece, false, WHITE_LONG_CASTLE);
					}	
					else if (!blackKingCastled)
					{
						if (square == INITIAL_BLACK_KING_SQUARE &&
							squarePiece[INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE] == ROCK &&
							squareSide[INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE] == BLACK &&
							squareSide[BLACK_SHORT_CASTLE_PASS1] == EMPTY &&
							squareSide[BLACK_SHORT_CASTLE_PASS2] == EMPTY &&
							!IsSquareInAttack(INITIAL_BLACK_KING_SQUARE, BLACK) &&
							!IsSquareInAttack(BLACK_SHORT_CASTLE_PASS1, BLACK) &&
							!IsSquareInAttack(BLACK_SHORT_CASTLE_PASS2, BLACK))
							TestAndAddMove(square, BLACK_SHORT_CASTLE_PASS2, movingPiece, false, BLACK_SHORT_CASTLE);
						
						if (square == INITIAL_BLACK_KING_SQUARE &&
							squarePiece[INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE] == ROCK &&
							squareSide[INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE] == BLACK &&
							squareSide[BLACK_LONG_CASTLE_PASS1] == EMPTY &&
							squareSide[BLACK_LONG_CASTLE_PASS2] == EMPTY && 
							squareSide[BLACK_LONG_CASTLE_PASS3] == EMPTY &&
							!IsSquareInAttack(INITIAL_BLACK_KING_SQUARE, BLACK) &&
							!IsSquareInAttack(BLACK_LONG_CASTLE_PASS1, BLACK) &&
							!IsSquareInAttack(BLACK_LONG_CASTLE_PASS2, BLACK)) //PASS3 is not tested for check
							TestAndAddMove(square, BLACK_LONG_CASTLE_PASS2, movingPiece, false, BLACK_LONG_CASTLE);
					
					}
				
				}
				else if (movingPiece == KNIGHT)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						
						//Los casilleros de piezas amigas son ignorados completamente
						//pero el testeo sigue en el resto de los casilleros
						if (moveSquareSide == EMPTY) 
							TestAndAddMove(square, moveSquare, movingPiece, false, -1);
						//capturo una pieza
						else if (moveSquareSide != sideToMove)
							TestAndAddMove(square, moveSquare, movingPiece, true, -1);
					}
				}
				else //(QUEEN)
				{
					for (direction=0; direction<8; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = eightDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							//se deslizo a un casillero vacio
							else if (moveSquareSide == EMPTY) 
								TestAndAddMove(square, moveSquare, movingPiece, false, -1);
							//capturo una pieza
							else 
							{
								TestAndAddMove(square, moveSquare, movingPiece, true, -1);
								break;
							}
						}
					}
				}
			
			}
		}
	}
}

void Engine::GenerateAllCaptures()
{
	//Clear previus arrays values. 
	//Te trick is copy blank moves in the directions of memory where previous moves are stored.
	memcpy((void*)legalCapture, (void*)blankLegalCapture, sizeof(Move)*CAPTURE_MOVE_COUNT);
	memcpy((void*)legalPromoteAndCapture, (void*)blankLegalPromoteAndCapture, sizeof(Move)*PROMOTE_AND_CAPTURE_MOVE_COUNT);

	legalCaptureCount = 0;
    legalPromoteAndCaptureCount = 0;

	int column;

	int movingPiece;
	int pawnCaptureLeftSquare;
	int pawnCaptureRightSquare;

	int direction;
	int directionSquare;
	int moveSquare;
	int moveSquareSide;

	legalNonCapture[0];

	for (int square=0; square<64; square++)
    {
		if (sideToMove == squareSide[square])
        {
			movingPiece = squarePiece[square];
			column = columnNumbers[square];
		
			if (movingPiece == PAWN)
            {
				if (sideToMove == WHITE)
                {
					//Camtures
                    pawnCaptureLeftSquare = square + 7;
                    if (column != 0 && squareSide[pawnCaptureLeftSquare] == BLACK)
                        TestAndAddMove(square, pawnCaptureLeftSquare, PAWN, true, -1);

                    pawnCaptureRightSquare = square + 9;
                    if (column != 7 && squareSide[pawnCaptureRightSquare] == BLACK)
                        TestAndAddMove(square, pawnCaptureRightSquare, PAWN, true, -1);
				}
				else
				{					
					//Camtures
                    pawnCaptureLeftSquare = square - 7;
                    if (column != 7 && squareSide[pawnCaptureLeftSquare] == WHITE)
                        TestAndAddMove(square, pawnCaptureLeftSquare, PAWN, true, -1);

                    pawnCaptureRightSquare = square - 9;
                    if (column != 0 && squareSide[pawnCaptureRightSquare] == WHITE)
                        TestAndAddMove(square, pawnCaptureRightSquare, PAWN, true, -1);
				}
			}
			else
			{
				
				GenerateNoPawnDirections(movingPiece, square);
				if (movingPiece == ROCK || movingPiece == BISHOP)
				{
					for (direction=0; direction<4; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = fourDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							else if (moveSquareSide != EMPTY)
							{
								TestAndAddMove(square, moveSquare, movingPiece, true, -1);
								break;
							}
						}
					}
				}
				else if (movingPiece == KING)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						if (moveSquareSide != sideToMove && moveSquareSide != EMPTY)
							TestAndAddMove(square, moveSquare, movingPiece, true, -1);
					}			
				}
				else if (movingPiece == KNIGHT)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						if (moveSquareSide != sideToMove && moveSquareSide != EMPTY)
							TestAndAddMove(square, moveSquare, movingPiece, true, -1);
					}
				}
				else //(QUEEN)
				{
					for (direction=0; direction<8; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = eightDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];
							
							if (moveSquareSide == sideToMove) 
								break;
							else if (moveSquareSide != EMPTY)
							{
								TestAndAddMove(square, moveSquare, movingPiece, true, -1);
								break;
							}
						}
					}
				}
			
			}
		}
	}
}

void Engine::TestAndAddMove(int from, int to, int movingPiece, bool capture, int castleMoveCode)
{
	int rank = rankNumbers[to];
	bool promote = ((rank == 0) || (rank == 7));

	if (movingPiece == PAWN)
    {
		if (capture && promote)
		{
			AddToLegalPromoteAndCaptureMoveList(from, to, PAWN, QUEEN);
			AddToLegalPromoteAndCaptureMoveList(from, to, PAWN, ROCK);
			AddToLegalPromoteAndCaptureMoveList(from, to, PAWN, BISHOP);
			AddToLegalPromoteAndCaptureMoveList(from, to, PAWN, KNIGHT);
		}
		else if (promote)
		{
			AddToLegalPromoteMoveList(from, to, PAWN, QUEEN);
			AddToLegalPromoteMoveList(from, to, PAWN, ROCK);
			AddToLegalPromoteMoveList(from, to, PAWN, KNIGHT);
			AddToLegalPromoteMoveList(from, to, PAWN, BISHOP);	
		}
		else if (capture)
			AddToLegalCaptureMoveList(from, to, PAWN); 
		else
			AddToLegalNonCaptureMoveList(from, to, PAWN, -1);
	}
	else
	{
		if (capture)
			AddToLegalCaptureMoveList(from, to, movingPiece);	
        else 
			AddToLegalNonCaptureMoveList(from, to, movingPiece, castleMoveCode);
	}
}

void Engine::AddToLegalCaptureMoveList(int from, int to, int movingPiece)
{
	legalCapture[legalCaptureCount].From = from;
	legalCapture[legalCaptureCount].To = to;
	legalCapture[legalCaptureCount].MovingSide = sideToMove;
	legalCapture[legalCaptureCount].MovingPiece = movingPiece;
	legalCapture[legalCaptureCount].DestinationPiece = squarePiece[to];
	legalCapture[legalCaptureCount].DestinationSide = squareSide[to];
	legalCapture[legalCaptureCount].Capture = true;
	legalCapture[legalCaptureCount].Check = false;
	legalCapture[legalCaptureCount].CastleMoveCode = -1;
	legalCaptureCount++;
}
void Engine::AddToLegalNonCaptureMoveList(int from, int to, int movingPiece, int castleMoveCode)
{
	legalNonCapture[legalNonCaptureCount].From = from;
	legalNonCapture[legalNonCaptureCount].To = to;
	legalNonCapture[legalNonCaptureCount].MovingSide = sideToMove;
	legalNonCapture[legalNonCaptureCount].MovingPiece = movingPiece;
	legalNonCapture[legalNonCaptureCount].DestinationPiece = squarePiece[to];
	legalNonCapture[legalNonCaptureCount].DestinationSide = squareSide[to];
	legalNonCapture[legalNonCaptureCount].Capture = false;
	legalNonCapture[legalNonCaptureCount].Check = false;
	legalNonCapture[legalNonCaptureCount].CastleMoveCode = castleMoveCode;
	legalNonCaptureCount++;

}
void Engine::AddToLegalPromoteAndCaptureMoveList(int from, int to, int movingPiece, int pieceToPromote)
{
	legalPromoteAndCapture[legalPromoteAndCaptureCount].From = from;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].To = to;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].MovingSide = sideToMove;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].MovingPiece = movingPiece;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].DestinationPiece = squarePiece[to];
	legalPromoteAndCapture[legalPromoteAndCaptureCount].DestinationSide = squareSide[to];
	legalPromoteAndCapture[legalPromoteAndCaptureCount].Capture = true;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].Check = false;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].Promote = true;
	legalPromoteAndCapture[legalPromoteAndCaptureCount].PieceToPromote = pieceToPromote;
	legalCapture[legalCaptureCount].CastleMoveCode = -1;
	legalPromoteAndCaptureCount++;
}
void Engine::AddToLegalPromoteMoveList(int from, int to, int movingPiece, int pieceToPromote)
{
	legalPromote[legalPromoteCount].From = from;
	legalPromote[legalPromoteCount].To = to;
	legalPromote[legalPromoteCount].MovingSide = sideToMove;
	legalPromote[legalPromoteCount].MovingPiece = movingPiece;
	legalPromote[legalPromoteCount].DestinationPiece = squarePiece[to];
	legalPromote[legalPromoteCount].DestinationSide = squareSide[to];
	legalPromote[legalPromoteCount].Capture = false;
	legalPromote[legalPromoteCount].Check = false;
	legalPromote[legalPromoteCount].Promote = true;
	legalPromote[legalPromoteCount].PieceToPromote = pieceToPromote;
	legalCapture[legalCaptureCount].CastleMoveCode = -1;
	legalPromoteCount++;
}

//Implements MVV/LVA for captures and promotes-captures
GeneratedMoves* Engine::GenerateAllSortedMoves()
{
	//GeneratedMoves* moves = new GeneratedMoves();
	GeneratedMoves* moves = (GeneratedMoves*)malloc(sizeof(GeneratedMoves));

	GenerateAllMoves();
	moves->MovesCount = legalPromoteAndCaptureCount+legalPromoteCount+legalCaptureCount+legalNonCaptureCount;
	
	moves->AllLegalMoves[legalPromoteAndCaptureCount+legalPromoteCount+legalCaptureCount+legalNonCaptureCount];

	qsort(legalPromoteAndCapture, legalPromoteAndCaptureCount, sizeof(Move), MoveComparer);
	qsort(legalPromote, legalPromoteCount, sizeof(Move), MoveComparer);
	qsort(legalCapture, legalCaptureCount, sizeof(Move), MoveComparer);
	qsort(legalNonCapture, legalNonCaptureCount, sizeof(Move), MoveComparer);


	memcpy((void*)moves->AllLegalMoves, (void*)legalPromoteAndCapture, sizeof(Move)*legalPromoteAndCaptureCount);
	memcpy((void*)(moves->AllLegalMoves+legalPromoteAndCaptureCount), (void*)legalPromote, sizeof(Move)*legalPromoteCount);
	memcpy((void*)(moves->AllLegalMoves+legalPromoteAndCaptureCount+legalPromoteCount), (void*)legalCapture, sizeof(Move)*legalCaptureCount);
	memcpy((void*)(moves->AllLegalMoves+legalPromoteAndCaptureCount+legalPromoteCount+legalCaptureCount), (void*)legalNonCapture, sizeof(Move)*legalNonCaptureCount);

	moves->MovesCount = legalPromoteAndCaptureCount+legalPromoteCount+legalCaptureCount+legalNonCaptureCount;

	return moves;
}

//Implements MVV/LVA for captures and promotes-captures
GeneratedMoves* Engine::GenerateAllSortedCaptures()
{
	//GeneratedMoves* moves = new GeneratedMoves();
	GeneratedMoves* captures = (GeneratedMoves*)malloc(sizeof(GeneratedMoves));

	GenerateAllCaptures();
	captures->AllLegalMoves[legalPromoteAndCaptureCount+legalCaptureCount];
	captures->MovesCount = legalPromoteAndCaptureCount+legalCaptureCount;

	int i;
	int allLegalMoveCount = 0;

	qsort(legalPromoteAndCapture, legalPromoteAndCaptureCount, sizeof(Move), MoveComparer);
	qsort(legalCapture, legalCaptureCount, sizeof(Move), MoveComparer);
	
	for (i=0; i<legalPromoteAndCaptureCount; i++)
	{
		captures->AllLegalMoves[allLegalMoveCount] = legalPromoteAndCapture[i];
		allLegalMoveCount++;
	}
	for (i=0; i<legalCaptureCount; i++)
	{
		captures->AllLegalMoves[allLegalMoveCount] = legalCapture[i];
		allLegalMoveCount++;
	}
	return captures;
}

int Engine::MoveComparer (const void* a, const void* b)
{
	Move movea = *(const Move *)a;
	Move moveb = *(const Move *)b;
	int moveAScore = movea.GetPreevaluateScore();
	int moveBScore = moveb.GetPreevaluateScore();
	
	if (moveAScore > moveBScore)
		return -1;
	else if (moveAScore < moveBScore)
		return 1;
	else
		return 0;
}

//////////////////////////////////////////////////////////////////////
// Board - Moke move / take back move
//////////////////////////////////////////////////////////////////////

void Engine::MakeMove(Move move)
{
    int movingSide = squareSide[move.From];
    int piece = squarePiece[move.From];
  	
    squareSide[move.From] = EMPTY;
    squareSide[move.To] = movingSide;

    if (move.Promote)
        squarePiece[move.To] = move.PieceToPromote;
    else
        squarePiece[move.To] = piece;

    //Update kings position
    if (move.MovingPiece == KING)
    {
        if (sideToMove == WHITE)
            whiteKingSquare = move.To;
        else
            blackKingSquare = move.To;
    }

	if (move.CastleMoveCode != -1)
	{
		//If caste, move the rock.
		if (move.CastleMoveCode == WHITE_SHORT_CASTLE)
		{
			squareSide[INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE] = EMPTY;
			squareSide[WHITE_SHORT_CASTLE_PASS1] = WHITE;
			squarePiece[WHITE_SHORT_CASTLE_PASS1] = ROCK;
			whiteKingCastled = true;
		}
		else if (move.CastleMoveCode == BLACK_SHORT_CASTLE)
		{
			squareSide[INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE] = EMPTY;
			squareSide[BLACK_SHORT_CASTLE_PASS1] = BLACK;
			squarePiece[BLACK_SHORT_CASTLE_PASS1] = ROCK;
			blackKingCastled = true;
		}
		else if (move.CastleMoveCode == WHITE_LONG_CASTLE)
		{
			squareSide[INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE] = EMPTY;
			squareSide[WHITE_LONG_CASTLE_PASS1] = WHITE;
			squarePiece[WHITE_LONG_CASTLE_PASS1] = ROCK;
			whiteKingCastled = true;
		}
		else if (move.CastleMoveCode == BLACK_LONG_CASTLE)
		{
			squareSide[INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE] = EMPTY;
			squareSide[BLACK_LONG_CASTLE_PASS1] = BLACK;
			squarePiece[BLACK_LONG_CASTLE_PASS1] = ROCK;
			blackKingCastled = true;
		} 
	}

	moveCount++;
	sideToMove = (sideToMove == WHITE) ? BLACK : WHITE;
}

void Engine::TakeBackMove(Move move)
{
    squareSide[move.From] = move.MovingSide;
    squarePiece[move.From] = move.MovingPiece;

    squareSide[move.To] = move.DestinationSide;
    squarePiece[move.To] = move.DestinationPiece;

    //Update kings position
    if (move.MovingPiece == KING)
    {
        if (move.MovingSide == WHITE)
            whiteKingSquare = move.From;
        else
            blackKingSquare = move.From;
    }

	sideToMove = (sideToMove == WHITE) ? BLACK : WHITE;
	if (move.CastleMoveCode == -1)
		return;

	//If caste, move the rock.
	if (move.CastleMoveCode == WHITE_SHORT_CASTLE)
	{
		squareSide[INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE] = WHITE;
		squareSide[WHITE_SHORT_CASTLE_PASS1] = EMPTY;
		squarePiece[INITIAL_WHITE_SHORT_CASTLE_ROCK_SQUARE] = ROCK;
		whiteKingCastled = false;
	}
	else if (move.CastleMoveCode == BLACK_SHORT_CASTLE)
	{
		squareSide[INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE] = BLACK;
		squareSide[BLACK_SHORT_CASTLE_PASS1] = EMPTY;
		squarePiece[INITIAL_BLACK_SHORT_CASTLE_ROCK_SQUARE] = ROCK;
		blackKingCastled = false;
	}
	else if (move.CastleMoveCode == WHITE_LONG_CASTLE)
	{
		squareSide[INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE] = WHITE;
		squareSide[WHITE_LONG_CASTLE_PASS1] = EMPTY;
		squarePiece[INITIAL_WHITE_LONG_CASTLE_ROCK_SQUARE] = ROCK;
		whiteKingCastled = false;
	}
	else if (move.CastleMoveCode == BLACK_LONG_CASTLE)
	{
		squareSide[INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE] = BLACK;
		squareSide[BLACK_LONG_CASTLE_PASS1] = EMPTY;
		squarePiece[INITIAL_BLACK_LONG_CASTLE_ROCK_SQUARE] = ROCK;
		blackKingCastled = false;
	}   
	moveCount--;
}

//////////////////////////////////////////////////////////////////////
// Board - Check testing
//////////////////////////////////////////////////////////////////////

bool Engine::InCheck(int side)
{
	bool retVal = false;
	if (side == BLACK)
	{
		retVal = IsSquareInAttack(blackKingSquare, BLACK);	
	}
	else if (side == WHITE)
	{
		retVal = IsSquareInAttack(whiteKingSquare, WHITE);
	}
	return retVal;
}

//////////////////////////////////////////////////////////////////////
// Search
//////////////////////////////////////////////////////////////////////

Move Engine::GetBestMove(int depth)
{
	Move move;
    //2147483647  -2147483647
	//int moveScore = FindCheckMate(sideToMove, depth+2, depth+2, INFINITE_NEGATIVE, INFINITE_POSITIVE, &move);
       
	//if (moveScore < INFINITE_POSITIVE)
	int	moveScore = AlphaBetaNegamax(true, depth, INFINITE_NEGATIVE, INFINITE_POSITIVE, &move);
	
	if (moveScore <= INFINITE_NEGATIVE )
    {
		if (move.To == move.From)
		{
			if (sideToMove == WHITE)
				GameStatus = WHITE_CHECKMATED;
			else
				GameStatus = BLACK_CHECKMATED;
		}
		else
		{	
			if (sideToMove == WHITE)
				GameStatus = WHITE_RESIGN;
			else 
				GameStatus = BLACK_RESIGN;
		}
		return move;
    }
	else if (moveScore >= INFINITE_POSITIVE)
	{
		GameStatus = CHECKMATE_FOUND;
		return move;
	}

    return move;	
}

//Same search algorithm but only finds checkmate
int Engine::FindCheckMate(int sideToEval, int depth, int initialDepth, int alpha, int beta, Move* bestMove)
{
	int bestScore = INFINITE_NEGATIVE;
    int moveScore = bestScore;
   
	GeneratedMoves* moves = GenerateAllSortedMoves();

	for (int i=0; i<moves->MovesCount; i++)
    {
		Move move = moves->AllLegalMoves[i];
		
		//Give max score to king capture to discard this move search.
        //This force to dont put own king in check!
        if (move.Capture && move.DestinationPiece == KING && depth != initialDepth)
		{
			free(moves);
            return INFINITE_POSITIVE;
		}

		//White was made a move
		MakeMove(move);
				
		//Test only check moves.
		if (InCheck(sideToMove) || sideToEval == sideToMove)	
		{				
			if (depth == 0) 
				//Dont eval, only test checkmate
				moveScore = 0;
			else
				moveScore = -FindCheckMate(sideToEval, depth-1, depth, -beta, -alpha, bestMove);

			if (moveScore > bestScore)
			{
				bestScore = moveScore;

				//Save the move only if is a player move.
				if (depth == initialDepth )
				{		
					bestMove->From = move.From;
					bestMove->To = move.To;
					bestMove->MovingPiece = move.MovingPiece;
					bestMove->MovingSide = move.MovingSide;
					bestMove->DestinationPiece = move.DestinationPiece;
					bestMove->DestinationSide = move.DestinationSide;
					bestMove->Capture = move.Capture;
					bestMove->Check = move.Check;
					bestMove->Promote = move.Promote;
					bestMove->PieceToPromote = move.PieceToPromote;
					bestMove->Score = move.Score;
					bestMove->CastleMoveCode = move.CastleMoveCode;
				}
			}
			if (bestScore > alpha)
				alpha = bestScore;
			/*if (alpha >= beta)
			{
				free(moves);
				return alpha;
			}*/
		}

		TakeBackMove(move);
    }
	
	free(moves);
    return bestScore;
}

//Implements a negamax search whit alfa beta cuttof and some modifications for speed-up checkmate.
int Engine::AlphaBetaNegamax(bool firstPass, int depth, int alpha, int beta, Move* bestMove)
{
	int bestScore = INFINITE_NEGATIVE;
    int moveScore = bestScore;
	bool foundPV = false;
   
	if (depth == 0) 
        //Eval the position for the current side to move
        return Quiescent(MAX_QPLY, alpha, beta);

	// if in check, search deeper
    bool inCheck = InCheck(sideToMove);
	bool hasMoves = false;
	int recursiveDepth = depth;

	//Search extension: In check
	if(inCheck)
	{
		//Al elegir el best move esto me caga!!!!!
		depth++;
	}

	GeneratedMoves* moves = GenerateAllSortedMoves();

	for (int i=0; i<moves->MovesCount; i++)
    {
		Move move = moves->AllLegalMoves[i];
        MakeMove(move);
		if (!InCheck(move.MovingSide))
		{		  
			hasMoves = true;
			if (foundPV) 
			{
				moveScore = -AlphaBetaNegamax(firstPass? !firstPass:firstPass, depth-1, -alpha - 1, -alpha, bestMove);
				if ((moveScore > alpha) && (moveScore < beta)) // Check for failure.
					moveScore = -AlphaBetaNegamax(firstPass? !firstPass:firstPass, depth-1, -beta, -alpha, bestMove);
			} 
			else
				moveScore = -AlphaBetaNegamax(firstPass? !firstPass:firstPass, depth-1, -beta, -alpha, bestMove);
		
			TakeBackMove(move);

			if (moveScore > bestScore )
			{
				bestScore = moveScore;

				//Save the move only if is a player move.
				if (firstPass)
				{				
					bestMove->From = move.From;
					bestMove->To = move.To;
					bestMove->MovingPiece = move.MovingPiece;
					bestMove->MovingSide = move.MovingSide;
					bestMove->DestinationPiece = move.DestinationPiece;
					bestMove->DestinationSide = move.DestinationSide;
					bestMove->Capture = move.Capture;
					bestMove->Check = move.Check;
					bestMove->Promote = move.Promote;
					bestMove->PieceToPromote = move.PieceToPromote;
					bestMove->Score = move.Score;
					bestMove->CastleMoveCode = move.CastleMoveCode;
				}
			}
			if (bestScore > alpha)
			{
				alpha = bestScore;
				foundPV = true;
			}
			if (alpha >= beta)
			{
				free(moves);
				return alpha;
			}
		}
		else
			TakeBackMove(move);
    }

	if (!hasMoves)
	{	
		free(moves);
		return inCheck? INFINITE_NEGATIVE : 0; 
	}
	
	free(moves);
    return bestScore;
}

int Engine::Quiescent(int quiesDepht, int alpha, int beta)
{
	if (quiesDepht == 0)
		return Evaluate();

    int value = Evaluate();
    if (value >= beta)
        return beta;
    if (value > alpha)
		alpha = value;

    GeneratedMoves* captures = GenerateAllSortedCaptures();
	for (int i=0; i<captures->MovesCount; i++)
    {
		Move capture = captures->AllLegalMoves[i];
		MakeMove(capture);
		value = -Quiescent(quiesDepht-1, -beta, -alpha);
		TakeBackMove(capture);
		
		if (value >= beta)
		{
			free(captures);
            return beta;
		}
        if (value > alpha)
            alpha = value;
	}
	free(captures);
    return alpha;
}


//////////////////////////////////////////////////////////////////////
// Evaluation
//////////////////////////////////////////////////////////////////////
int Engine::Evaluate()
{
	int square, column, rank, r=0, offsetColumn, offsetRank, attacksCount=0;
	int pieceInSquare=0, sideInSquare=2;
	
	int whiteScore=0, blackScore=0;
	
	//Use 12*12 arrays to represent board to avoid border testing
	int whitePawn[12][12]={0}, blackPawn[12][12]={0};
	int whitePiece[12][12]={0}, blackPiece[12][12]={0};
	int whitePawnByColumn[12]={0}, blackPawnByColumn[12]={0};

	int whiteNoPawnMaterial=0, blackNoPawnMaterial=0;
	int whitePawnMaterial=0, blackPawnMaterial=0;
	int whitePawnScore=0, blackPawnScore=0;
	int whitePieceDevelopment=0, blackPieceDevelopment=0;

	int whiteBoardControlScore=0, blackBoardControlScore=0;

	bool oppening = false, ending=false;
	
	int pawnScoreOppening[]  = 
	{	
		0,   0,   0,   0,   0,   0,   0,   0,
		0,   0,   0, -10, -10,   0,   0,   0,
		1,   2,   3,   4,   4,   3,   2,   1,
		0,   3,   4,   8,   8,   4,   3,   0,
		0,   1,   5,  10,  10,   5,   1,   0,
		0,   0,   3,   8,   8,   3,   0,   0,
		0,   0,   0,   0,   0,   0,   0,   0,
		0,   0,   0,   0,   0,   0,   0,   0,
	};

	int pawnScoreEnding[]  = 
	{	
		0,   0,   0,   0,   0,   0,   0,   0,
		0,   0,   0, -10, -10,   0,   0,   0,
		3,   5,   6,   8,   8,   6,   5,   3,
		6,  10,  12,  16,  16,  12,  10,   6,
		9,  15,  18,  24,  24,  18,  15,   9,
	   12,  20,  24,  32,  32,  24,  20,  12,
	   15,  25,  30,  32,  32,  30,  25,  15,
	   15,  25,  30,   0,   0,  30,  25,  15,
	};

	int knightScore[]  = 
	{	
		0,   5,   7,   7,   7,   7,   5,   0,
		5,   7,  10,  10,  10,  10,   7,   5,
		7,  10,  20,  20,  20,  20,  10,   7,
		7,  10,  20,  30,  30,  20,  10,   7,
		7,  10,  20,  30,  30,  20,  10,   7,
		7,  10,  20,  20,  20,  20,  10,   7,
	    5,   7,  10,  10,  10,  10,   7,   5,
	    0,   5,   7,   7,   7,   7,   5,   0,
	};

	int bishopScore[]  = 
	{	
	   14,  15,  16,  17,  17,  16,  15,   14,
	   15,  16,  17,  18,  18,  17,  16,   15,
	   16,  17,  19,  20,  20,  19,  17,   16,
	   17,  18,  20,  22,  22,  20,  18,   17,
	   17,  18,  20,  22,  22,  20,  18,   17,
	   16,  17,  19,  20,  20,  19,  17,   16,
	   15,  16,  17,  18,  18,  17,  16,   15,
	   14,  15,  16,  17,  17,  16,  15,   14,
	};

	int kingScore[]  = 
	{	
	   -5,  -2,  20,   0,   0,   0,  20,  -2,
	  -10,  -5,  -5, -20, -20, -20,  -5,  -5,
	  -10, -10, -15, -22, -22, -15, -10, -10,
	  -15, -20, -22, -24, -24, -22, -20, -15,
	  -15, -20, -22, -24, -24, -22, -20, -15,
	  -30, -30, -30, -30, -30, -30, -30, -30,
	  -30, -30, -30, -30, -30, -30, -30, -30,
	  -30, -30, -30, -30, -30, -30, -30, -30,
	};

	int kingEndingScore[]  = 
	{	
	   0,   5,   6,   7,   7,   6,   5,    0,
	   5,   7,   9,  10,  10,   9,   7,    5,
	   6,   9,  15,  20,  20,  15,   9,    6,
	   7,  10,  20,  36,  36,  20,  10,    7,
	   7,  10,  20,  36,  36,  20,  10,    7,
	   6,   9,  15,  20,  20,  15,   9,    6,
	   5,   7,   9,  10,  10,   9,   7,    5,
	   0,   5,   6,   7,   7,   6,   5,    4,
	};

	int flip[] = 
	{
		56,  57,  58,  59,  60,  61,  62,  63,
		48,  49,  50,  51,  52,  53,  54,  55,
		40,  41,  42,  43,  44,  45,  46,  47,
		32,  33,  34,  35,  36,  37,  38,  39,
		24,  25,  26,  27,  28,  29,  30,  31,
		16,  17,  18,  19,  20,  21,  22,  23,
		 8,   9,  10,  11,  12,  13,  14,  15,
		 0,   1,   2,   3,   4,   5,   6,   7
	};

	memcpy(attackSquares, blankAttackSquares, sizeof(int)*64);
	
	//First pass compute base material and create piece array.
	for (square=0; square<64; square++)
    {		
        pieceInSquare = squarePiece[square];
		sideInSquare = squareSide[square];

		column = columnNumbers[square];
		rank = rankNumbers[square];
		
		offsetColumn = column+2;
		offsetRank = rank+2;

		if (sideInSquare == EMPTY)
			TestAttacks(square);
	
		if (sideInSquare == WHITE)
		{	
			if (pieceInSquare == PAWN)
			{
				whitePawnMaterial += pieceValue[PAWN];
				whitePawn[offsetRank][offsetColumn] = 1;
				whitePawnByColumn[offsetColumn]++;
			}
			else
				whiteNoPawnMaterial += pieceValue[pieceInSquare];

			whitePiece[offsetRank][offsetColumn] = 1;
		}

		else if (sideInSquare == BLACK)
		{
			if (pieceInSquare == PAWN)
			{
				blackPawnMaterial += pieceValue[PAWN];
				blackPawn[offsetRank][offsetColumn] = 1;
				blackPawnByColumn[offsetColumn]++;
			}
			else
				blackNoPawnMaterial += pieceValue[pieceInSquare];

			blackPiece[offsetRank][offsetColumn] = 1;
		}
	}

	oppening = (moveCount<=OPENING_MOVES);

	if (blackNoPawnMaterial>whiteNoPawnMaterial)
		ending = (whiteNoPawnMaterial<ENDING_NO_PAWN_SCORE);
	else
		ending = (blackNoPawnMaterial<ENDING_NO_PAWN_SCORE);
	

	//Second pass test pieces bonusses and penalties
	for (square=0; square<64; square++)
	{		
		pieceInSquare = squarePiece[square];
		sideInSquare = squareSide[square];

		column = columnNumbers[square];
		rank = rankNumbers[square];

		offsetColumn = column+2;
		offsetRank = rank+2;


		if (sideInSquare==EMPTY)
		{
			if (attackSquares[square]>0)
				whiteBoardControlScore += attackSquares[square] * EMPTY_SQUARE_ATTACK_BONUS;
			else if (attackSquares[square]<0)
				blackBoardControlScore += -attackSquares[square] + EMPTY_SQUARE_ATTACK_BONUS;
		}


		if (sideInSquare == WHITE)
		{	
			if (pieceInSquare == PAWN)
			{
				//ISOLATE PAWN: No pawn in left column and no pawn in right column
				if (whitePawnByColumn[offsetColumn-1] == 0 && whitePawnByColumn[offsetColumn+1] == 0)
					whitePawnScore += isolatedPawnPenalty[column];
				
				//BACKWARD PAWN: Behind the pawns of the same color on the adjacent files and that cannot be advanced with the support of another pawn.
				else if ((whitePawn[offsetRank+1][offsetColumn-1] == 1 || whitePawn[offsetRank+1][offsetColumn+1] == 1) &&
						 (whitePawn[offsetRank][offsetColumn-1] == 0 && whitePawn[offsetRank][offsetColumn+1] == 0))
				{	
					whitePawnScore += BACKWARD_PAWN_PENALTY;
				
					//PENALTY FOR EACH OPOSITE ATTACK IS APPLIED
					whitePawnScore += BACKWARD_PAWN_ATTACKED_PENALTY * CountAttacks(square, BLACK);

					//PENALTY FOR A BACKWARD PAWN IN A HALF OPEN ENEMY FILE IS APPLIED
					if (blackPawnByColumn[offsetColumn] == 0) 
						whitePawnScore += BACKWARD_PAWN_IN_HALF_OPEN_FILE_PENALTY;

				}

				//DOUBLED PAWN: Another pawn are in same column. Doubled penalty computes for ALL doubled pawns.
				if (whitePawnByColumn[offsetColumn] > 1)
					whitePawnScore += DOUBLED_PAWN_PENALTY;


				//PAWN ADVANCEMENT
				whitePawnScore += ending? pawnScoreEnding[square]:pawnScoreOppening[square];
				
				//BLOCKED e-d pawns
				//d-pawn
				if (whitePawn[3][5]==1 && (whitePiece[4][5]==1 || blackPiece[4][5]==1))
					whitePawnScore += BLOCKED_D_E_PAWN_PENALTY;
				//e-pawn
				if (whitePawn[3][6]==1 && (whitePiece[4][6]==1 || blackPiece[4][6]==1))
					whitePawnScore += BLOCKED_D_E_PAWN_PENALTY;

				//PASED PAWN: No opponent pawn in the same column or adyacent columns fron the next rank
				//to the last rank.
				bool passedPawn = true;
				for (r=rank+1; r<=7; r++)
				{
					//Check if an opposite pawn are in the path of the pawn or in adyacent columns
					if (blackPawn[r+2][offsetColumn]==1 || blackPawn[r+2][offsetColumn-1]==1 || blackPawn[r+2][offsetColumn+1]==1)
					{
						passedPawn = false;
						break;
					}
				}
				if (passedPawn)
				{
					whitePawnScore += (rank-1)*40 + 15; //215 max en rank 6, 55 on rank 2

					for (r=rank+1; r<=7 && rank!=2; r++)
					{
						if (IsSquareInAttack(r*8+column, WHITE) || squareSide[BLACK])
							whitePawnScore -= 10;
					}

					if (whiteNoPawnMaterial<ENDING_NO_PAWN_SCORE)
						whitePawnScore+=rank*15;	//295 on endgame max on rank 6 			
				}	

				//DISTANCE TO KING UP TO 2 SQUARES IS REWARDED
				if (GetDistance(square, whiteKingSquare)<=2)
					whitePawnScore += PAWN_CLOSED_TO_KING_BONUS;
			}
			else if (pieceInSquare == KNIGHT)
			{
				//Knight value based on square position
				whitePieceDevelopment += knightScore[square];
				
				//Knight penalty based on distance to king
				for(int s=0; s<64; s++)
				{
					if (squareSide[s] == BLACK && GetDistance(s, square)<=2)
						whitePieceDevelopment += oppening? KNIGHT_NEAR_OPPONENT_PIECE_BONUS_OPE:KNIGHT_NEAR_OPPONENT_PIECE_BONUS_END;	
					
				}
				whitePieceDevelopment += GetDistance(whiteKingSquare, square) * KNIGHT_FAR_FROM_KING_PENALTY;
			
				if (ending)
				{
					//There are no pawn than can capture this knight
					bool canBeDriveByPawn = false;
					for (r=rank+1; r<=7; r++)
					{
						if (blackPawn[r+2][offsetColumn-1]==1 || blackPawn[r+2][offsetColumn+1]==1)
						{
							canBeDriveByPawn = true;
							break;
						}
					}
					
					if(!canBeDriveByPawn)
						whitePieceDevelopment += KNIGHT_SECURE_FROM_ENEMY_PAWNS_BONUS;
				}
			}
			else if (pieceInSquare == BISHOP)
			{
				//BISHOP SCORE BASED ON ATTACK THE BODY OF PIECES AND NOT THE PAWN SKELETON
				whitePieceDevelopment += GetXRayScore(square, BISHOP, WHITE);
				
				//BISHOP IN ENDINGS ARE MORE VALUABLES
				whiteNoPawnMaterial += ending? BISHOP_EXTRA_MATERIAL_IN_ENDING : 0;
				whitePieceDevelopment += bishopScore[square];

				//BISHOP HAS A SCORE FOR ATTACK ADYACENT KING SQUARES
				int moveSquare;
				GenerateNoPawnDirections(KING, blackKingSquare);
				for (int directionSquare=0; directionSquare<8; directionSquare++)
				{
					moveSquare = oneDirection[directionSquare];
					if (moveSquare==INVALID_SQUARE)
						break;

					if (BishopAttackSquare(moveSquare, square))
						whitePieceDevelopment += BISHOP_ATTACK_ADYACENT_ENEMY_KING_BONUS;
				}

			}
			else if (pieceInSquare == ROCK)
			{
				whitePieceDevelopment += GetXRayScore(square, ROCK, WHITE);

				//ROC ARE IN A OPEN FILE
				if (whitePawnByColumn[offsetColumn]==0 && blackPawnByColumn[offsetColumn]==0)
					whitePieceDevelopment += ROCK_ON_HALF_OPEN_FILE_BONUS + ROCK_ON_FULL_OPEN_FILE_EXTRA_BONUS;
			
				//ROC ARE IN A HALF OPEN FILE
				else if (whitePawnByColumn[offsetColumn]==0)
					whitePieceDevelopment += ROCK_ON_HALF_OPEN_FILE_BONUS;
				
				//ROC ARE ON SEVENTH RANK
				if (rank==6)
					whitePieceDevelopment += ROCK_ON_SEVENTH_COLUMN_BONUS;

				if (!oppening)
				{
					if (GetDistance(blackKingSquare, square) > 4)
						whitePieceDevelopment += ROCK_TAXICAB_PENALTY;
				}

			}
			else if (pieceInSquare == QUEEN)
			{
				if (!oppening)
				{
					if (GetDistance(blackKingSquare, square) > 4)
						whitePieceDevelopment += QUEEN_TAXICAB_PENALTY;
				}
			}
			else if (pieceInSquare == KING)
			{
				//KING SCORE BY POSITION
				if (ending)
					whitePieceDevelopment += kingEndingScore[square];
				else
					whitePieceDevelopment += kingScore[square];

				//KING IS IN OPEN COLUMN
				if (whitePawnByColumn[offsetColumn] == 0)
					whitePieceDevelopment += KING_IN_OPEN_FILE_PENALTY;

				//THE NEAREST COLUMN TO BORDER OF THE KING IS OPEN OR HALF OPEN
				if (column < 4 && column>0)
				{
					if (whitePawnByColumn[offsetColumn-1] == 0)
						whitePieceDevelopment += KING_IN_OPEN_FILE_PENALTY;
				}
				else if(column<7)
				{
					if (whitePawnByColumn[offsetColumn+1] == 0)
						whitePieceDevelopment += KING_IN_OPEN_FILE_PENALTY;
				}

				//ADYACENT KING PAWNS
				int moveSquare;
				int kingPawnCount=0;
				GenerateNoPawnDirections(KING, whiteKingSquare);
				for (int directionSquare=0; directionSquare<8; directionSquare++)
				{
					moveSquare = oneDirection[directionSquare];
					if (moveSquare==INVALID_SQUARE)
						break;

					if (squarePiece[moveSquare] == PAWN && squareSide[moveSquare] == WHITE)
						kingPawnCount++;
				}

				//NO PAWNS ARE NEAR THE KING
				if (kingPawnCount==0)
					whitePieceDevelopment += KING_WITH_NO_PAWNS_PENALTY;

				//KING IS CASTLED
				if (whiteKingCastled)
					whitePieceDevelopment += KING_CASTLED_BONUS;

				//KING WS MOVED BEFORE CASTILNG
				if (whiteKingSquare != INITIAL_WHITE_KING_SQUARE && !whiteKingCastled)
					whitePieceDevelopment += KING_MOVED_BEFORE_CASTLING_PENALTY;

			}
		}

		else if (sideInSquare == BLACK)
		{
			if (pieceInSquare == PAWN)
			{
				if (blackPawnByColumn[offsetColumn-1] == 0 && blackPawnByColumn[offsetColumn+1] == 0)
					blackPawnScore += isolatedPawnPenalty[column];
				
				else if ((blackPawn[offsetRank-1][offsetColumn-1] == 1 || blackPawn[offsetRank-1][offsetColumn+1] == 1) &&
						 (blackPawn[offsetRank][offsetColumn-1] == 0 && blackPawn[offsetRank][offsetColumn+1] == 0))
				{
					blackPawnScore += BACKWARD_PAWN_PENALTY;
					blackPawnScore += BACKWARD_PAWN_ATTACKED_PENALTY	* CountAttacks(square, WHITE);

					if (whitePawnByColumn[offsetColumn] == 0) 
						blackPawnScore += BACKWARD_PAWN_IN_HALF_OPEN_FILE_PENALTY;
				}
				
				if (blackPawnByColumn[offsetColumn] > 1)
					blackPawnScore += DOUBLED_PAWN_PENALTY;

				blackPawnScore += ending? pawnScoreEnding[flip[square]]:pawnScoreOppening[flip[square]];
				
				//d-pawn
				if (blackPawn[8][5]==1 && (whitePiece[7][5]==1 || blackPiece[7][5]==1))
					blackPawnScore += BLOCKED_D_E_PAWN_PENALTY;
				//e-pawn
				if (blackPawn[8][6]==1 && (whitePiece[7][6]==1 || blackPiece[7][6]==1))
					blackPawnScore += BLOCKED_D_E_PAWN_PENALTY;

				bool passedPawn = true;
				for (r=rank-1; r>=0; r--)
				{
					if (whitePawn[r+2][offsetColumn]==1 || whitePawn[r+2][offsetColumn-1]==1 || whitePawn[r+2][offsetColumn+1]==1)
					{
						passedPawn = false;
						break;
					}
				}
				if (passedPawn)
				{
					blackPawnScore += (6-rank)*40 + 15; //215 max en rank 6, 55 on rank 2
					for (r=rank-1; r>=0; r--)
					{
						if (IsSquareInAttack(r*8+column, BLACK) || squareSide[WHITE])
							blackPawnScore -= 10;
					}
					if (blackNoPawnMaterial<ENDING_NO_PAWN_SCORE)
						blackPawnScore+= (7-rank)*15;	//295 on endgame max on rank 2 				
				}

				if (GetDistance(square, blackKingSquare)<=2)
					blackPawnScore += PAWN_CLOSED_TO_KING_BONUS;
			}
			else if (pieceInSquare == KNIGHT)
			{
				blackPieceDevelopment += knightScore[square];
				
				for(int s=0; s<64; s++)
				{
					if (squareSide[s] == WHITE && GetDistance(s, square)<=2)
						blackPieceDevelopment += oppening? KNIGHT_NEAR_OPPONENT_PIECE_BONUS_OPE:KNIGHT_NEAR_OPPONENT_PIECE_BONUS_END;	
					
				}
				blackPieceDevelopment += GetDistance(blackKingSquare, square) * KNIGHT_FAR_FROM_KING_PENALTY;
			
				if (ending)
				{
					bool canBeDriveByPawn = false;
					for (r=rank-1; r>=0; r--)
					{
						if (whitePawn[r+2][offsetColumn-1]==1 || whitePawn[r+2][offsetColumn+1]==1)
						{
							canBeDriveByPawn = true;
							break;
						}
					}
					
					if(!canBeDriveByPawn)
						blackPieceDevelopment += KNIGHT_SECURE_FROM_ENEMY_PAWNS_BONUS;
				}
			}
			else if (pieceInSquare == BISHOP)
			{
				blackPieceDevelopment += GetXRayScore(square, BISHOP, BLACK);
				blackNoPawnMaterial += ending? BISHOP_EXTRA_MATERIAL_IN_ENDING : 0;
				blackPieceDevelopment += bishopScore[square];

				int moveSquare;
				GenerateNoPawnDirections(KING, whiteKingSquare);
				for (int directionSquare=0; directionSquare<8; directionSquare++)
				{
					moveSquare = oneDirection[directionSquare];
					if (moveSquare==INVALID_SQUARE)
						break;

					if (BishopAttackSquare(moveSquare, square))
						blackPieceDevelopment += BISHOP_ATTACK_ADYACENT_ENEMY_KING_BONUS;
				}

			}
			else if (pieceInSquare == ROCK)
			{
				blackPieceDevelopment += GetXRayScore(square, ROCK, BLACK);

				//ROC ARE IN A HALF OPEN FILE
				if (whitePawnByColumn[offsetColumn]==0 && blackPawnByColumn[offsetColumn]==0)
					blackPieceDevelopment += ROCK_ON_HALF_OPEN_FILE_BONUS + ROCK_ON_FULL_OPEN_FILE_EXTRA_BONUS;
				else if (blackPawnByColumn[offsetColumn]==0)
					blackPieceDevelopment += ROCK_ON_HALF_OPEN_FILE_BONUS;
				
				if (!oppening)
				{
					if (GetDistance(whiteKingSquare, square) > 4)
						blackPieceDevelopment += ROCK_TAXICAB_PENALTY;
				}

				if (rank==2)
					blackPieceDevelopment += ROCK_ON_SEVENTH_COLUMN_BONUS;

			}
			else if (pieceInSquare == QUEEN)
			{
				if (!oppening)
				{
					if (GetDistance(whiteKingSquare, square) > 4)
						blackPieceDevelopment += QUEEN_TAXICAB_PENALTY;
				}
			}
			else if (pieceInSquare == KING)
			{
				if (ending)
					blackPieceDevelopment += kingEndingScore[flip[square]];
				else
					blackPieceDevelopment += kingScore[flip[square]];

				if (blackPawnByColumn[offsetColumn] == 0)
					blackPieceDevelopment += KING_IN_OPEN_FILE_PENALTY;

				if (column < 4 && column>0)
				{
					if (blackPawnByColumn[offsetColumn-1] == 0)
						blackPieceDevelopment += KING_IN_OPEN_FILE_PENALTY;
				}
				else if(column<7)
				{
					if (blackPawnByColumn[offsetColumn+1] == 0)
						blackPieceDevelopment += KING_IN_OPEN_FILE_PENALTY;
				}
				
				int moveSquare;
				int kingPawnCount=0;
				GenerateNoPawnDirections(KING, blackKingSquare);
				for (int directionSquare=0; directionSquare<8; directionSquare++)
				{
					moveSquare = oneDirection[directionSquare];
					if (moveSquare==INVALID_SQUARE)
						break;

					if (squarePiece[moveSquare] == PAWN && squareSide[moveSquare] == BLACK)
						kingPawnCount++;
				}

				if (kingPawnCount==0)
					blackPieceDevelopment += KING_WITH_NO_PAWNS_PENALTY;

				if (blackKingCastled)
					blackPieceDevelopment += KING_CASTLED_BONUS;

				if (blackKingSquare != INITIAL_BLACK_KING_SQUARE && !blackKingCastled)
					blackPieceDevelopment += KING_MOVED_BEFORE_CASTLING_PENALTY;

			}

		}
	}

	whiteScore += whiteBoardControlScore + whitePawnScore + whitePawnMaterial + whiteNoPawnMaterial + whitePieceDevelopment;
	blackScore += blackBoardControlScore + blackPawnScore + blackPawnMaterial + blackNoPawnMaterial + blackPieceDevelopment;

	return (sideToMove == WHITE) ? whiteScore - blackScore : blackScore - whiteScore;
}

int Engine::CountAttacks(int square, int attackerSide)
{
	int directionCounter = 0; //1 : 0, 2:45, 3:90, 4:135, 5:180, 6:225, 7:270, 8:315
	int moves, pieceInSquare, sideInSquare, direction, directionSquare, moveSquare;

	int attacksCount = 0;
	
	GenerateNoPawnDirections(QUEEN, square);
	for (direction=0; direction<8; direction++)
	{
		moves = 0;
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moves++;
			moveSquare = eightDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];
			
			if (sideInSquare == attackerSide)
            {
                if (pieceInSquare == KING && moves == 1) 
					attacksCount++;
				
                else if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
					attacksCount++;

				else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
					attacksCount++;

				else if (pieceInSquare == QUEEN)
					attacksCount++;

				else if (pieceInSquare == PAWN && moves == 1)
                {
                    if (sideInSquare == WHITE)
                    {
						if ((square == moveSquare + 7 && columnNumbers[moveSquare] != 0) ||
                            (square == moveSquare + 9 && columnNumbers[moveSquare] != 7))
							attacksCount++;
                    }
                    else
                    {
						if ((square == moveSquare - 7 && columnNumbers[moveSquare] != 7) ||
                           (square == moveSquare - 9 && columnNumbers[moveSquare] != 0))
							attacksCount++;
                    }
                }
                break;
            }
            else if (sideInSquare != EMPTY)
                break;
		}
	}
    
    GenerateNoPawnDirections(KNIGHT, square);
	for (directionSquare=0; directionSquare<8; directionSquare++)
	{
		moveSquare = oneDirection[directionSquare];
		if (moveSquare == INVALID_SQUARE)
			break;

		sideInSquare = squareSide[moveSquare];
		pieceInSquare = squarePiece[moveSquare];
		
		if (sideInSquare == attackerSide && pieceInSquare == KNIGHT)
			attacksCount++;
	}

	return attacksCount;
}

int Engine::GetDistance(int square1, int square2)
{
	int d1 = abs(columnNumbers[square1]-columnNumbers[square2]);
	int d2 = abs(rankNumbers[square1]-rankNumbers[square2]);

	if (d1>=d2)
		return d1;
	else
		return d2;
}

int Engine::GetXRayScore(int square, int piece, int sideOfPiece)
{
	//Bishop or rock compute xray score.
	int xRayScore = (piece==BISHOP)? -4:0;

	GenerateNoPawnDirections(piece, square);
   	int directionCounter = 0; 
	int pieceInSquare, sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<4; direction++)
	{
		directionCounter++;
		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moveSquare = fourDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE )
				break;

			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];
			
			if (sideInSquare == EMPTY)
				xRayScore += 2;
			else if (sideInSquare != sideOfPiece)
            {
                if (pieceInSquare == PAWN )
				{
					break;
				}
                else if (piece==BISHOP && (pieceInSquare == ROCK || pieceInSquare == QUEEN || pieceInSquare == KING))
                    xRayScore += 2 + BISHOP_ATTACK_MOST_VALUED_BONUS;
				else if (piece==ROCK && (pieceInSquare == QUEEN || pieceInSquare == KING)) 
                    xRayScore += 2 + ROCK_ATTACK_MOST_VALUED_BONUS;
				else
					xRayScore += 2;
            }
			else
				break;
		}
	}

	return xRayScore;
}

bool Engine::IsSquareInAttack(int square, int sideOfSquare)
{
	//Si testeo un casillero vacio el problema es que sideInSquare != sideForTest y testea los atyaques de las propias piezas 
	
	GenerateNoPawnDirections(QUEEN, square);
   	int directionCounter = 0; 
	int moves, pieceInSquare, sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<8; direction++)
	{
		moves = 0;
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moves++;
			moveSquare = eightDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];
			
			if (sideInSquare != EMPTY  && sideInSquare != sideOfSquare)
            {
                if (pieceInSquare == KING && moves == 1) //testeo rey
                    return true;
                //testeo bishop solo en diagonales
                else if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
                    return true;
                //testeo rock solo en filas y columnas
                else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
                    return true;
                //testeo queen en todas las direcciones
                else if (pieceInSquare == QUEEN)
                    return true;
                else if (pieceInSquare == PAWN && moves == 1)
                {
                    if (sideInSquare == WHITE)
                    {
                        if (((square == moveSquare + 7) && columnNumbers[moveSquare] != 0) ||
                            ((square == moveSquare + 9) && columnNumbers[moveSquare] != 7))
                            return true;
                    }
                    else
                    {
                        if (((square == moveSquare - 7) && columnNumbers[moveSquare] != 7) ||
                           ((square == moveSquare - 9) && columnNumbers[moveSquare] != 0))
                            return true;
                    }
                }
                break;
            }
            else if (sideInSquare != EMPTY)
                break;
		}
	}
    
    //Pruebo el rey como si fuera un caballo y busco otros caballos
    GenerateNoPawnDirections(KNIGHT, square);
	for (directionSquare=0; directionSquare<8; directionSquare++)
	{
		moveSquare = oneDirection[directionSquare];
		if (moveSquare == INVALID_SQUARE)
			break;

		sideInSquare = squareSide[moveSquare];
		pieceInSquare = squarePiece[moveSquare];
		
		if (sideInSquare != EMPTY && sideInSquare != sideOfSquare && pieceInSquare == KNIGHT) //testeo caballo
			return true;
	}

	return false;
}

bool Engine::BishopAttackSquare(int square, int attackerSquare)
{
	GenerateNoPawnDirections(BISHOP, square);
	int sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<4; direction++)
	{
		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moveSquare = fourDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = squareSide[moveSquare];

			if (moveSquare==attackerSquare)
				return true;
			else if (sideInSquare != EMPTY )
				break;
		}
	}

	return false;
}

void Engine::TestAttacks(int square)
{
	GenerateNoPawnDirections(QUEEN, square);
   	int directionCounter = 0; //1 : 0, 2:45, 3:90, 4:135, 5:180, 6:225, 7:270, 8:315
	int moves, pieceInSquare, sideInSquare, direction, directionSquare, moveSquare;

	int originalSquareSide = squareSide[square];
	int originalSquarePieceValue = pieceValue[squarePiece[square]];
	int attackFactor=1;

	for (direction=0; direction<8; direction++)
	{
		moves = 0;
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moves++;
			moveSquare = eightDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];
			
			if (sideInSquare != EMPTY)
            {
                if (pieceInSquare == KING && moves == 1) 
				{
				    attackSquares[square] += (sideInSquare == WHITE) ? 1: -1;
				}
                else if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
				{
					attackFactor = (originalSquarePieceValue > pieceValue[BISHOP] && originalSquareSide != sideInSquare && originalSquareSide != EMPTY)? 2:1;
					attackSquares[square] += (sideInSquare == WHITE) ? attackFactor: -attackFactor;
                }
				else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
                {
					attackFactor = (originalSquarePieceValue > pieceValue[BISHOP] && originalSquareSide != sideInSquare && originalSquareSide != EMPTY)? 2:1;
					attackSquares[square] += (sideInSquare == WHITE) ? attackFactor: -attackFactor;
				}
				else if (pieceInSquare == QUEEN)
				{
					attackSquares[square] += (sideInSquare == WHITE) ? 1 : -1;
                }
				else if (pieceInSquare == PAWN && moves == 1)
                {
                    if (sideInSquare == WHITE)
                    {
						attackFactor = (originalSquarePieceValue > pieceValue[PAWN] && originalSquareSide != sideInSquare && originalSquareSide != EMPTY)? 2:1;
					    if ((square == moveSquare + 7 && columnNumbers[moveSquare] != 0) ||
                            (square == moveSquare + 9) && columnNumbers[moveSquare] != 7)
                            attackSquares[square] += attackFactor;
                    }
                    else
                    {
						attackFactor = (originalSquarePieceValue > pieceValue[PAWN] && originalSquareSide != sideInSquare && originalSquareSide != EMPTY)? 2:1;
					    if ((square == moveSquare - 7 && columnNumbers[moveSquare] != 7) ||
                           (square == moveSquare - 9) && columnNumbers[moveSquare] != 0)
						   attackSquares[square] += -attackFactor;
                    }
                }
                break;
            }
            else if (sideInSquare != EMPTY)
                break;
		}
	}
    
    //Pruebo el rey como si fuera un caballo y busco otros caballos
    GenerateNoPawnDirections(KNIGHT, square);
	for (directionSquare=0; directionSquare<8; directionSquare++)
	{
		moveSquare = oneDirection[directionSquare];
		if (moveSquare == INVALID_SQUARE)
			break;

		sideInSquare = squareSide[moveSquare];
		pieceInSquare = squarePiece[moveSquare];
		
		attackFactor = (originalSquarePieceValue > pieceValue[KNIGHT] && originalSquareSide != sideInSquare && originalSquareSide != EMPTY)? 2:1;
		if (sideInSquare != EMPTY && pieceInSquare == KNIGHT) //testeo caballo
			attackSquares[square] += (sideInSquare == WHITE) ? attackFactor: -attackFactor;
	}
}

//////////////////////////////////////////////////////////////////////
// Testing
//////////////////////////////////////////////////////////////////////

void Engine::Test()
{
	//srand ( time(NULL) );

	long s = 0;
	int moveCount = 1;
    int sideCount = 0;
	int movesPerLine = 0;

	cout << endl;

	while (s < 100)
	{	
		Move move = GetBestMove(4);
		//Move move = GetRandomMove();

		//If return the null move, player lost or draw.
		if (move.To == move.From)
			break;
				
		MakeMove(move);
		//TakeBackMove(move);
		if (sideCount == 0)
        {
			cout << moveCount << ".";
			move.PrintMove();
        }
        else
			move.PrintMove();
                
		cout << " ";
                                      
        sideCount++;
		movesPerLine++;

        if (sideCount == 2)
        {
            sideCount = 0;
            moveCount++;
		}
		if (movesPerLine == 8)
		{
			cout << endl;
			movesPerLine = 0;
		}
		s++;
	}
	s = 0;
}


Move Engine::GetRandomMove()
{
	GeneratedMoves* moves = GenerateAllSortedMoves();
	int random = (rand()%moves->MovesCount);

	Move m = moves->AllLegalMoves[random];

	if (m.MovingSide != sideToMove)
	{
		int error = 1;
	}
    return m;
}

